/*

cvkbdarm.c - Coleco Keyboard ARM code

Modelled from lpc1114-blink-master sample.
Freddie Chopin, http://www.freddiechopin.info/

*/

/*...sincludes:0:*/
#include <stdint.h>
#include "inc/LPC11xx.h"
#include "inc/core_cm0.h"
#include "hdr/hdr_syscon.h"
#include "cvkbd.h"

/*...vinc\47\LPC11xx\46\h:0:*/
/*...vinc\47\core_cm0\46\h:0:*/
/*...vhdr\47\hdr_syscon\46\h:0:*/
/*...vcvkbd\46\h:0:*/
/*...e*/

/*...sdefines:0:*/
/* Input signals */
#define	PS2_CLK	11
#define	PS2_DAT 10
#define	K_WR_n   9
#define	SW0	 5
#define	SW1	 6
#define	KC       2
#define	KC_mask  7

/* Output signals */
#define	KD       0
#define	KD_mask  0xff
#define	LED0     8
#define	LED1     9

/* Scan code prefixes */
#define	PREFIX_E0 0x0100
#define	PREFIX_E1 0x0200
#define	PREFIX_F0 0x0400

/* Keyboard layouts */
#define	KL_UK 0
#define	KL_US 1
/*...e*/

/*...spll_start:0:*/
/*------------------------------------------------------------------------*//**
* \brief Starts the PLL.
* \details Configure and enable PLL to achieve some frequency with some
* crystal. Before the speed change flash access time is configured via
* flash_access_time(). Main oscillator is configured and started. PLL
* parameters m and p are based on function parameters. The PLL is configured,
* started and selected as the main clock. AHB clock divider is set to 1.
*
* \param [in] crystal is the frequency of the crystal resonator connected to
* the LPC1114 chip.
* \param [in] frequency is the desired target frequency after enabling the PLL
*
* \return real frequency that was set
*//*-------------------------------------------------------------------------*/

static uint32_t pll_start(uint32_t crystal, uint32_t frequency)
{
	uint32_t m, p = 0, fcco;

	// set flash access time for >= 40MHz
	uint32_t flashcfg_register;
	flashcfg_register = FLASHCFG;
	flashcfg_register &= ~(FLASHCFG_FLASHTIM_mask << FLASHCFG_FLASHTIM_bit);
	flashcfg_register |=  (FLASHCFG_FLASHTIM_3CLK << FLASHCFG_FLASHTIM_bit);
	FLASHCFG = flashcfg_register;

	// SYSOSCCTRL_FREQRANGE should be 0 for crystals in range 1 - 20MHz
	// SYSOSCCTRL_FREQRANGE should be 1 for crystals in range 15 - 25MHz
	if (crystal < 17500000)					// divide the ranges on 17.5MHz then
		LPC_SYSCON->SYSOSCCTRL = 0;			// "lower speed" crystals
	else
		LPC_SYSCON->SYSOSCCTRL = SYSOSCCTRL_FREQRANGE;	// "higher speed" crystals

	LPC_SYSCON->PDRUNCFG &= ~PDRUNCFG_SYSOSC_PD;	// power-up main oscillator

	LPC_SYSCON->SYSPLLCLKSEL = SYSPLLCLKSEL_SEL_IRC;	// select main oscillator as the input clock for PLL
	LPC_SYSCON->SYSPLLCLKUEN = 0;			// confirm the change of PLL input clock by toggling the...
	LPC_SYSCON->SYSPLLCLKUEN = SYSPLLUEN_ENA;	// ...ENA bit in LPC_SYSCON->SYSPLLCLKUEN register

	// calculate PLL parameters
	m = frequency / crystal;				// M is the PLL multiplier
	fcco = m * crystal * 2;					// FCCO is the internal PLL frequency

	frequency = crystal * m;

	while (fcco < 156000000)
	{
		fcco *= 2;
		p++;								// find P which gives FCCO in the allowed range (over 156MHz)
	}

	LPC_SYSCON->SYSPLLCTRL = ((m - 1) << SYSPLLCTRL_MSEL_bit) | (p << SYSPLLCTRL_PSEL_bit);	// configure PLL
	LPC_SYSCON->PDRUNCFG &= ~PDRUNCFG_SYSPLL_PD; // power-up PLL

	while (!(LPC_SYSCON->SYSPLLSTAT & SYSPLLSTAT_LOCK));	// wait for PLL lock

	LPC_SYSCON->MAINCLKSEL = MAINCLKSEL_SEL_PLLOUT;	// select PLL output as the main clock
	LPC_SYSCON->MAINCLKUEN = 0;				// confirm the change of main clock by toggling the...
	LPC_SYSCON->MAINCLKUEN = MAINCLKUEN_ENA;	// ...ENA bit in LPC_SYSCON->MAINCLKUEN register

	LPC_SYSCON->SYSAHBCLKDIV = 1;			// set AHB clock divider to 1

	return frequency;
}
/*...e*/
/*...sPIOINT0_IRQHandler:0:*/
/* Internal to the ISR */
static uint8_t  n_clk    = 0;
static uint16_t prefixes = 0;
static uint8_t  scan;
static uint8_t  parity;

/* Communication between ISR and background loop */
static int scan_producer = 0;
static int scan_consumer = 0;
#define	N_SCANS 0x100
static uint16_t scans[N_SCANS];
static uint8_t  kc = 0xff;

static uint8_t n_errs = 0;

/* Note, observed interrupt latency is ~1us, and duration is ~5us. */

void PIOINT0_IRQHandler(void) __attribute__ ((interrupt));
void PIOINT0_IRQHandler(void)
	{
	if ( LPC_GPIO0->MIS & (1<<PS2_CLK) )
		{
		/* We've configured falling edges to cause interrupts.
		   It looks like the odd rising edges causes interrupts.
		   However, this could well be noise, so try to filter it. */
		uint8_t ps2_clk = (uint8_t) ( ( LPC_GPIO0->MASKED_ACCESS[1<<PS2_CLK] ) >> PS2_CLK );
		if ( ps2_clk == 0 )
			{
			/* We want to decode the scan code as quickly as possible,
			   so as not to lose transitions. So we do all the processing in the ISR. */
			uint8_t ps2_dat = (uint8_t) ( ( LPC_GPIO0->MASKED_ACCESS[1<<PS2_DAT] ) >> PS2_DAT );
			if ( n_clk == 0 )
				/* expecting start bit */
				{
				if ( ps2_dat == 0 )
					{
					n_clk  = 1;
					parity = 0;
					}
				}
			else if ( n_clk <= 8 )
				/* expecting data bit n_clk */
				{
				scan = (uint8_t) ( (scan>>1) | (ps2_dat<<7) );
				parity ^= ps2_dat;
				++n_clk;
				}
			else if ( n_clk == 9 )
				/* expecting parity */
				{
				if ( parity != ps2_dat )
					/* parity is ok */
					n_clk = 10;
				else
					/* parity is not ok, start again */
					{
					++n_errs;
					n_clk = 0;
					}
				}
			else
				/* expecting stop bit */
				{
				if ( ps2_dat )
					/* stop bit ok */
					{
					if ( scan == 0xe0 )
						prefixes |= PREFIX_E0;
					else if ( scan == 0xe1 )
						prefixes |= PREFIX_E1;
					else if ( scan == 0xf0 )
						prefixes |= PREFIX_F0;
					else
						/* Make new scan code available to main loop.
						   The code to convert to keycodes takes
						   a while, so we won't do it here. */
						{
						if ( scan_producer - scan_consumer < N_SCANS )
							scans[scan_producer++%N_SCANS] = prefixes | scan;
						prefixes = 0;
						}
					}
				else
					/* bad stop bit, start again */
					++n_errs;
				n_clk = 0;
				}
			}
		LPC_GPIO0->IC = (1<<PS2_CLK);
		}
	if ( LPC_GPIO0->MIS & (1<<K_WR_n) )
		{
		/* Read command and make available to main loop. */
		kc = (uint8_t) (LPC_GPIO0->MASKED_ACCESS[KC_mask<<KC] >> KC);
		LPC_GPIO0->IC = (1<<K_WR_n);
		}
	}
/*...e*/
/*...smap:0:*/
/* These tables are initialised for UK keyboards */

static uint8_t scan_to_keycode[] =
	{
/* 	x0	x1	x2	x3	x4	x5	x6	x7	x8	x9	xa	xb	xc	xd	xe	xf	*/
	0,	KD_F9,	0,	KD_F5,	KD_F3,	KD_F1,	KD_F2,	KD_F12,	0,	KD_F10,	KD_F8,	KD_F6,	KD_F4,	KD_TAB,	'`',	0,	/* 0x */
	0,	KD_LALT,KD_LSH,	0,	KD_LCTL,'q',	'1',	0,	0,	0,	'z',	's',	'a',	'w',	'2',	0,	/* 1x */
	0,	'c',	'x',	'd',	'e',	'4',	'3',	0,	0,	' ',	'v',	'f',	't',	'r',	'5',	0,	/* 2x */
	0,	'n',	'b',	'h',	'g',	'y',	'6',	0,	0,	0,	'm',	'j',	'u',	'7',	'8',	0,	/* 3x */
	0,	',',	'k',	'i',	'o',	'0',	'9',	0,	0,	'.',	'/',	'l',	';',	'p',	'-',	0,	/* 4x */
	0,	0,	'\'',	0,	'[',	'=',	0,	0,	KD_CAPL,KD_RSH,	KD_RET,	']',	0,	'#',	0,	0,	/* 5x */
	0,	'\\',	0,	0,	0,	0,	KD_BS,	0,	0,	KD_NMED,0,	KD_NML,	KD_NMHM,0,	0,	0,	/* 6x */
	KD_NMIN,KD_NMDL,KD_NMD,	KD_NMC,	KD_NMR,	KD_NMU,	KD_ESC,	KD_NMLK,KD_F11,	KD_NMP,	KD_NMPD,KD_NMM,	KD_NMT,	KD_NMPU,KD_SCLK,0,	/* 7x */
	0,	0,	0,	KD_F7,	KD_PSC,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	/* 8x */
	};

typedef struct
	{
	uint8_t scan;
	uint8_t keycode;
	} SCANKEY;

static const SCANKEY scankey_e0[] =
	{
		{ 0x7c, KD_PSC  },
		{ 0x70, KD_IN   },
		{ 0x6c, KD_HM   },
		{ 0x7d, KD_PU   },
		{ 0x7e, KD_PAUS },
		{ 0x77, KD_NMLK },
		{ 0x4a, KD_NMDV },
		{ 0x71, KD_DL   },
		{ 0x69, KD_ED   },
		{ 0x7a, KD_PD   },
		{ 0x75, KD_U    },
		{ 0x5a, KD_NMET },
		{ 0x1f, KD_LWIN },
		{ 0x11, KD_RALT },
		{ 0x27, KD_RWIN },
		{ 0x2f, KD_MENU },
		{ 0x14, KD_RCTL },
		{ 0x6b, KD_L    },
		{ 0x72, KD_D    },
		{ 0x74, KD_R    },
	};

static void set_layout(uint8_t layout)
	{
	switch ( layout )
		{
		case KL_UK:
			break;
		case KL_US:
			scan_to_keycode[0x5d] = '\\';
			scan_to_keycode[0x61] = 0;
			break;
		}
	}

static uint8_t map_uncooked(uint16_t scan)
	{
	if ( scan & PREFIX_E0 )
		{
		int i;
		for ( i = 0; i < (int)(sizeof(scankey_e0)/sizeof(scankey_e0[0])); i++ )
			if ( scankey_e0[i].scan == (uint8_t) scan )
				return scankey_e0[i].keycode;
		}
	else if ( scan & PREFIX_E1 )
		{
		if ( (scan&0xff) == 0x77 )
			return KD_PAUS;
		}
	else
		{
		if ( scan < sizeof(scan_to_keycode) &&
		     scan_to_keycode[scan] != 0 )
			return scan_to_keycode[scan];
		}
	return KD_UNKN;
	}
/*...e*/
/*...smain:0:*/
/* Unconsumed keycodes */
static int keycode_producer = 0;
static int keycode_consumer = 0;
#define	N_KEYCODES 0x100
static uint8_t keycodes[N_KEYCODES];

#define	M_LSH	0x01
#define	M_RSH	0x02
#define	M_LCTL	0x04
#define	M_RCTL	0x08
#define	M_LALT	0x10
#define	M_RALT	0x20
#define	M_CAPL	0x40
#define	M_NUML	0x80
static uint8_t modifiers = 0;

/*...sis_first_32:0:*/
static int is_first_32(uint8_t keycode)
	{
	return ( keycode >= 'a' && keycode <= 'z' ) ||
	       ( keycode >= '@' && keycode <= '_' );
	}
/*...e*/

/*...sapply_cap:0:*/
static uint8_t apply_cap(uint8_t keycode)
	{
	if ( keycode >= 'a' && keycode <= 'z' )
		return keycode-'a'+'A';
	return keycode;
	}
/*...e*/
/*...sapply_sh:0:*/
static uint8_t apply_sh(uint8_t keycode, uint8_t layout)
	{
	if ( keycode >= 'a' && keycode <= 'z' )
		return keycode-'a'+'A';

	switch ( layout )
		{
		case KL_UK:
			switch ( keycode )
				{
				case '2':	return '"';
				case '3':	return KD_POUN;
				case '\'':	return '@';
				case '#':	return '~';
				}
			break;
		case KL_US:
			switch ( keycode )
				{
				case '2':	return '@';
				case '3':	return '#';
				case '\'':	return '"';
				case '`':	return '~';
				}
			break;
		}

	switch ( keycode )
		{
		case '1':	return '!';
		case '4':	return '$';
		case '5':	return '%';
		case '6':	return '^';
		case '7':	return '&';
		case '8':	return '*';
		case '9':	return '(';
		case '0':	return ')';
		case '-':	return '_';
		case '=':	return '+';
		case '[':	return '{';
		case ']':	return '}';
		case ';':	return ':';
		case ',':	return '<';
		case '.':	return '>';
		case '/':	return '?';
		case '\\':	return '|';
		}

	return keycode;
	}
/*...e*/
/*...sapply_ctl:0:*/
static uint8_t apply_ctl(uint8_t keycode)
	{
	if ( keycode == KD_BS )
		return KD_CBS;
	return is_first_32(keycode) ? (keycode&0x1f) : keycode;
	}
/*...e*/
/*...sapply_alt:0:*/
static uint8_t apply_alt(uint8_t keycode)
	{
	return is_first_32(keycode) ? ((keycode&0x1f)|0x80) : keycode;
	}
/*...e*/
/*...sapply_num:0:*/
static uint8_t apply_num(uint8_t keycode)
	{
	switch ( keycode )
		{
		case KD_NMDL:	return '.';
		case KD_NMIN:	return '0';
		case KD_NMED:	return '1';
		case KD_NMD:	return '2';
		case KD_NMPD:	return '3';
		case KD_NML:	return '4';
		case KD_NMC:	return '5';
		case KD_NMR:	return '6';
		case KD_NMHM:	return '7';
		case KD_NMU:	return '8';
		case KD_NMPU:	return '9';
		}
	return keycode;
	}
/*...e*/

/* Remember how we cooked each key */
static uint8_t cooked[0x100];

int main(void)
	{
	uint8_t layout;

	/* Run at 50MHz */
	pll_start(12000000, 50000000);

	/* Enable clock for IO configuration block */
	LPC_SYSCON->SYSAHBCLKCTRL |= SYSAHBCLKCTRL_IOCON;

	/* Ensure all GPIO0_x are inputs */
	LPC_GPIO0->DIR = 0;
	/* All interrupts are edge sensitive */
	LPC_GPIO0->IS  = 0;
	/* Don't trigger on both edges */
	LPC_GPIO0->IBE = 0;
	/* Trigger K_WR_n on rising edge */
	LPC_GPIO0->IEV = (1<<K_WR_n);
	/* No interrupts for now */
	LPC_GPIO0->IE  = 0;

	/* Enable all the outputs on GPIO1 */
	LPC_GPIO1->DIR |= ( ((uint32_t)KD_mask<<KD) |
	                    ((uint32_t)1<<LED0)     |
	                    ((uint32_t)1<<LED1)     );

	/* Ensure GPIO1_0..3 actually work as GPIO, not special functions.
	   The other GPIOs seem to initialise to a sensible state. */
	LPC_IOCON->R_PIO1_0     = 0xd1;
	LPC_IOCON->R_PIO1_1     = 0xd1;
	LPC_IOCON->R_PIO1_2     = 0xd1;
	LPC_IOCON->SWDIO_PIO1_3 = 0xd1;

	/* Sample the keyboard layout and configure accordingly */
	layout = (uint8_t) (LPC_GPIO0->MASKED_ACCESS[1<<SW0] >> SW0);
	set_layout(layout);

	/* Enable interrupts */
	LPC_GPIO0->IE  = ( (1<<PS2_CLK) | (1<<K_WR_n) );

	/* Something other than hi-Z, so that we can be detected by Z80 */
	LPC_GPIO1->MASKED_ACCESS[(uint32_t)KD_mask<<KD] = 0;

	NVIC_EnableIRQ(EINT0_IRQn);

	for ( ;; )
		{
		uint8_t sw1 = (uint8_t) (LPC_GPIO0->MASKED_ACCESS[1<<SW1] >> SW1);
		if ( sw1 )
			/* Debugging */
			{
			LPC_GPIO1->MASKED_ACCESS[1<<LED0] =
				( ( ( modifiers & (M_LSH|M_RSH|M_LCTL|M_RCTL|M_LALT|M_RALT) ) != 0 ) << LED0 );
			LPC_GPIO1->MASKED_ACCESS[1<<LED1] = ((n_errs&1)<<LED1);
			}
		else
			/* Reflect modifiers and caps-lock state in LEDs */
			{
			LPC_GPIO1->MASKED_ACCESS[1<<LED0] =
				( ( ( modifiers & M_CAPL ) != 0 ) << LED0 );
			LPC_GPIO1->MASKED_ACCESS[1<<LED1] =
				( ( ( modifiers & M_NUML ) != 0 ) << LED1 );
			}
		/* Only keep interrupts disabled for a very short time,
		   just long enough to get our next item of work. */
		__disable_irq();
		if ( scan_consumer < scan_producer &&
		     keycode_producer - keycode_consumer <= N_KEYCODES-2 )
			{
			uint16_t scan_in = scans[scan_consumer++%N_SCANS];
			__enable_irq();
/*...stranslate scan_in to keycode:24:*/
{
uint8_t keycode = map_uncooked(scan_in & ~PREFIX_F0);
if ( scan_in & PREFIX_F0 )
	{
	keycodes[keycode_producer++%N_KEYCODES] = KD_RELE;
	switch ( keycode )
		{
		case KD_LSH:	modifiers &= ~M_LSH;	break;
		case KD_RSH:	modifiers &= ~M_RSH;	break;
		case KD_LCTL:	modifiers &= ~M_LCTL;	break;
		case KD_RCTL:	modifiers &= ~M_RCTL;	break;
		case KD_LALT:	modifiers &= ~M_LALT;	break;
		case KD_RALT:	modifiers &= ~M_RALT;	break;
		case KD_CAPL:				break;
		case KD_NMLK:				break;
		default:
			/* we don't care what modifiers are in force now.
			   we care how we previously cooked the key. */
			keycode = cooked[keycode];
			break;
		}
	}
else
	{
	switch ( keycode )
		{
		case KD_LSH:	modifiers |= M_LSH;	break;
		case KD_RSH:	modifiers |= M_RSH;	break;
		case KD_LCTL:	modifiers |= M_LCTL;	break;
		case KD_RCTL:	modifiers |= M_RCTL;	break;
		case KD_LALT:	modifiers |= M_LALT;	break;
		case KD_RALT:	modifiers |= M_RALT;	break;
		case KD_CAPL:	modifiers ^= M_CAPL;	break;
		case KD_NMLK:	modifiers ^= M_NUML;	break;
		default:
			{
			uint8_t keycode_cooked = keycode;
			if ( modifiers & (M_LSH|M_RSH) )
				keycode_cooked = apply_sh(keycode_cooked, layout);
			else if ( modifiers & M_CAPL )
				keycode_cooked = apply_cap(keycode_cooked);
			if ( modifiers & (M_LCTL|M_RCTL) )
				keycode_cooked = apply_ctl(keycode_cooked);
			else if ( modifiers & (M_LALT|M_RALT) )
				keycode_cooked = apply_alt(keycode_cooked);
			if ( modifiers & M_NUML )
				keycode_cooked = apply_num(keycode_cooked);
			cooked[keycode] = keycode_cooked;
			keycode = keycode_cooked;
			}
			break;
		}
	}
keycodes[keycode_producer++%N_KEYCODES] = keycode;
if ( LPC_GPIO1->MASKED_ACCESS[(uint32_t)KD_mask<<KD] == (((uint32_t)KD_NONE)<<KD) )
	LPC_GPIO1->MASKED_ACCESS[(uint32_t)KD_mask<<KD] = (((uint32_t)keycodes[keycode_consumer%N_KEYCODES]) << KD);
}
/*...e*/
			}
		else if ( kc != 0xff )
			{
			uint8_t kc_in = kc;
			kc = 0xff;
			__enable_irq();
/*...srespond to command from Z80:24:*/
switch ( kc_in )
	{
	case KC_INIT:
		__disable_irq();
		scan_producer = scan_consumer = 0;
		__enable_irq();
		keycode_producer = keycode_consumer = 0;
		LPC_GPIO1->MASKED_ACCESS[(uint32_t)KD_mask<<KD] = (((uint32_t)KD_INIT)<<KD);
		break;
	case KC_PEEK:
		{
		uint8_t kd = ( keycode_consumer == keycode_producer )
			? KD_NONE
			: keycodes[keycode_consumer&0xff];
		LPC_GPIO1->MASKED_ACCESS[(uint32_t)KD_mask<<KD] = (((uint32_t)kd)<<KD);
		}
		break;
	case KC_NEXT:
		{
		if ( keycode_consumer != keycode_producer )
			++keycode_consumer;
		LPC_GPIO1->MASKED_ACCESS[(uint32_t)KD_mask<<KD] = (((uint32_t)KD_NEXT)<<KD);
		}
		break;
	}
/*...e*/
			}
		else
			__enable_irq();
		}
	}
/*...e*/
